﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mediapackage-vod/MediaPackageVodRequest.h>
#include <aws/mediapackage-vod/MediaPackageVod_EXPORTS.h>

#include <utility>

namespace Aws {
namespace MediaPackageVod {
namespace Model {

/**
 * A new MediaPackage VOD Asset configuration.<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreateAssetRequest">AWS
 * API Reference</a></p>
 */
class CreateAssetRequest : public MediaPackageVodRequest {
 public:
  AWS_MEDIAPACKAGEVOD_API CreateAssetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAsset"; }

  AWS_MEDIAPACKAGEVOD_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * The unique identifier for the Asset.
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  CreateAssetRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The ID of the PackagingGroup for the Asset.
   */
  inline const Aws::String& GetPackagingGroupId() const { return m_packagingGroupId; }
  inline bool PackagingGroupIdHasBeenSet() const { return m_packagingGroupIdHasBeenSet; }
  template <typename PackagingGroupIdT = Aws::String>
  void SetPackagingGroupId(PackagingGroupIdT&& value) {
    m_packagingGroupIdHasBeenSet = true;
    m_packagingGroupId = std::forward<PackagingGroupIdT>(value);
  }
  template <typename PackagingGroupIdT = Aws::String>
  CreateAssetRequest& WithPackagingGroupId(PackagingGroupIdT&& value) {
    SetPackagingGroupId(std::forward<PackagingGroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The resource ID to include in SPEKE key requests.
   */
  inline const Aws::String& GetResourceId() const { return m_resourceId; }
  inline bool ResourceIdHasBeenSet() const { return m_resourceIdHasBeenSet; }
  template <typename ResourceIdT = Aws::String>
  void SetResourceId(ResourceIdT&& value) {
    m_resourceIdHasBeenSet = true;
    m_resourceId = std::forward<ResourceIdT>(value);
  }
  template <typename ResourceIdT = Aws::String>
  CreateAssetRequest& WithResourceId(ResourceIdT&& value) {
    SetResourceId(std::forward<ResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * ARN of the source object in S3.
   */
  inline const Aws::String& GetSourceArn() const { return m_sourceArn; }
  inline bool SourceArnHasBeenSet() const { return m_sourceArnHasBeenSet; }
  template <typename SourceArnT = Aws::String>
  void SetSourceArn(SourceArnT&& value) {
    m_sourceArnHasBeenSet = true;
    m_sourceArn = std::forward<SourceArnT>(value);
  }
  template <typename SourceArnT = Aws::String>
  CreateAssetRequest& WithSourceArn(SourceArnT&& value) {
    SetSourceArn(std::forward<SourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The IAM role ARN used to access the source S3 bucket.
   */
  inline const Aws::String& GetSourceRoleArn() const { return m_sourceRoleArn; }
  inline bool SourceRoleArnHasBeenSet() const { return m_sourceRoleArnHasBeenSet; }
  template <typename SourceRoleArnT = Aws::String>
  void SetSourceRoleArn(SourceRoleArnT&& value) {
    m_sourceRoleArnHasBeenSet = true;
    m_sourceRoleArn = std::forward<SourceRoleArnT>(value);
  }
  template <typename SourceRoleArnT = Aws::String>
  CreateAssetRequest& WithSourceRoleArn(SourceRoleArnT&& value) {
    SetSourceRoleArn(std::forward<SourceRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateAssetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateAssetRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;
  bool m_idHasBeenSet = false;

  Aws::String m_packagingGroupId;
  bool m_packagingGroupIdHasBeenSet = false;

  Aws::String m_resourceId;
  bool m_resourceIdHasBeenSet = false;

  Aws::String m_sourceArn;
  bool m_sourceArnHasBeenSet = false;

  Aws::String m_sourceRoleArn;
  bool m_sourceRoleArnHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaPackageVod
}  // namespace Aws
